<?php

namespace Elementor;

use Chakavak_Controls_Elementor;
use WP_Query;

defined('ABSPATH') or exit;

/**
 * Widget Name: Posts Carousel
 */
class chakavak_Posts_Carousel extends Widget_Base
{
    const NAME = 'chk_posts_carousel';
    const CATEGORY = 'chakavak_widgets';

    public function get_name()
    {
        return self::NAME;
    }

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_title()
    {
        return __('Posts Carousel', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-posts-carousel';
    }

    public function get_keywords()
    {
        return ['blog', 'post', 'post carousel', 'carousel', esc_html__('posts carousel', 'chakavak')];
    }

    public function get_style_depends()
    {
        return [
            'chk-posts',
            'swiper-bundle',
        ];
    }

    public function get_script_depends()
    {
        return [
            'swiper-bundle',
            'chk-widgets',
        ];
    }

    protected function register_controls()
    {
        /*General Settings*/
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('General settings', 'chakavak'),
            ]
        );

        $this->add_control(
            'post_style',
            [
                'label' => esc_html__('Display Style', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 's1',
                'options' => [
                    's1' => esc_html__('Style 1', 'chakavak'),
                    's2' => esc_html__('Style 2', 'chakavak'),
                ]
            ]
        );

        Chakavak_Controls_Elementor::element_tag($this, 'title_tag', esc_html__('Title tag', 'chakavak'), 'h3');

        $this->add_control(
            'query_heading',
            [
                'label' => esc_html__('Query', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'post_cats',
            [
                'label' => __('Select Categories', 'chakavak'),
                'type' => Controls_Manager::SELECT2,
                'options' => chakavak_get_categories_list('category'),
                'multiple' => true,
                'label_block' => true,
                'placeholder' => __('All Categories', 'chakavak'),
            ]
        );

        $this->add_control(
            'post_ids',
            [
                'label' => __('Select Posts', 'chakavak'),
                'type' => Controls_Manager::SELECT2,
                'default' => '',
                'multiple' => true,
                'label_block' => true,
                'options' => chakavak_get_posts('post'),
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label' => __('Show Number portfolio', 'chakavak'),
                'type' => Controls_Manager::NUMBER,
                'default' => '9',
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'DESC',
                'options' => [
                    'DESC' => esc_html__('Descending', 'chakavak'),
                    'ASC' => esc_html__('Ascending', 'chakavak'),
                ]
            ]
        );

        $this->add_control(
            'show_excerpt',
            [
                'label' => esc_html__('Excerpt', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'excerpt_length',
            [
                'label' => esc_html__('Excerpt Text Length', 'chakavak'),
                //'label_block' => true,
                'type' => Controls_Manager::NUMBER,
                'default' => '160',
                'condition' => [
                    'show_excerpt' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'post_metas_heading',
            [
                'label' => esc_html__('Post Metas', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'show_cat',
            [
                'label' => esc_html__('Category', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'show_date',
            [
                'label' => esc_html__('Date', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'show_author',
            [
                'label' => esc_html__('Author Name', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'show_comments',
            [
                'label' => esc_html__('Comment Count', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'target_link',
            [
                'label' => esc_html__('Target open link', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => '_self',
                'options' => [
                    '_blank' => esc_html__('New page', 'chakavak'),
                    '_self' => esc_html__('self page', 'chakavak'),
                ]
            ]
        );

        $this->add_control(
            'button_text',
            [
                'label' => __('Read more label', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'default' => __('View article', 'chakavak'),
                'condition' => [
                    'post_style' => 's2',
                ],
            ]
        );

        $this->end_controls_section();

        Chakavak_Controls_Elementor::add_control_carousel($this, false);

        $this->start_controls_section(
            'item_style',
            [
                'label' => esc_html__('Item', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_radius',
            [
                'label' => esc_html__('Border Radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'item_background',
                'label' => esc_html__('Card Background', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .chk-post-item',
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Item Background', 'chakavak'),
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'label' => __('Item Border', 'chakavak'),
                'exclude' => ['color'], //remove border color
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width' => [
                        'default' => [
                            'top' => '1',
                            'right' => '1',
                            'bottom' => '1',
                            'left' => '1',
                            'isLinked' => false,
                        ],
                    ],
                ],
                'selector' => '{{WRAPPER}} .chk-post-item',
            ]
        );

        //tabs

        $this->start_controls_tabs('item_tabs');

        $this->start_controls_tab(
            'item_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'border_color',
            [
                'label' => __('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => 'var(--chk-c-gray-200)',
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item' => 'border-color:{{VALUE}}'
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_shadow_normal',
                'label' => esc_html__('Item Shadow', 'chakavak'),
                'selector' => '{{WRAPPER}} .chk-post-item',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'item_hover',
            [
                'label' => __('Hover', 'chakavak'),
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_shadow_hover',
                'label' => esc_html__('Item Shadow', 'chakavak'),
                'selector' => '{{WRAPPER}} .chk-post-item:hover',
                'fields_options' => [
                    'box_shadow_type' =>
                        [
                            'default' => 'yes'
                        ],
                    'box_shadow' => [
                        'default' =>
                            [
                                'horizontal' => 2,
                                'vertical' => 3,
                                'blur' => 10,
                                'spread' => 0,
                                'color' => 'rgba(0,0,0,0.1)'
                            ]
                    ]
                ],
            ]
        );

        $this->add_control(
            'item_border_color_hover',
            [
                'label' => __('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item:hover' => 'border-color:{{VALUE}}'
                ]
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        //image card style
        $this->start_controls_section(
            'image_style',
            [
                'label' => esc_html__('Image', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'show_image',
            [
                'label' => esc_html__('Show Image', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        Chakavak_Controls_Elementor::image_size($this, 'image', esc_html__('Select image Size', 'chakavak'), 'chk_blog', 'none', ['show_image' => 'yes']);

        $this->add_responsive_control(
            'image_radius',
            [
                'label' => esc_html__('Border Radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__thumbnail' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .chk-post-item.s2 .chk-post-item__thumbnail:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'show_image' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'image_hover_effect',
            [
                'label' => esc_html__('Hover Effect', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'none',
                'options' => [
                    'none' => esc_html__('None', 'chakavak'),
                    'zoom' => esc_html__('Zoom', 'chakavak'),
                    'overly' => esc_html__('Overlay', 'chakavak'),
                    'grayscale' => esc_html__('Grayscale', 'chakavak')
                ],
                'condition' => [
                    'show_image' => 'yes'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'image_shadow',
                'label' => esc_html__('Image shadow', 'chakavak'),
                'selector' => '{{WRAPPER}} .chk-post-item__thumbnail',
                'condition' => [
                    'show_image' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'content_styling',
            [
                'label' => esc_html__('Content', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label' => __('Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'chakavak'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'chakavak'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'chakavak'),
                        'icon' => 'eicon-text-align-right',
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__content' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'content_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        /* title */
        $this->add_control(
            'heading_title',
            [
                'label' => __('Title', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'exclude' => [
                    'text_transform',
                ],
                'selector' => '{{WRAPPER}} .chk-post-item__title',
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => esc_html__('Space Bottom', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_inline_limit',
            [
                'label' => esc_html__('Title inline?', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label' => esc_html__('Title color hover', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__title a:is(:hover,:focus)' => 'color: {{VALUE}};',
                ],
            ]
        );

        /* Excerpt */
        $this->add_control(
            'heading_excerpt',
            [
                'label' => __('Excerpt', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_excerpt' => 'yes'
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'excerpt_typography',
                'selector' => '{{WRAPPER}} .chk-post-item__excerpt',
                'condition' => [
                    'show_excerpt' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'excerpt_color',
            [
                'label' => esc_html__('Excerpt color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__excerpt' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_excerpt' => 'yes'
                ]
            ]
        );

        /* Meta */
        $this->add_control(
            'heading_meta',
            [
                'label' => esc_html__('Meta', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'meta_margin_top',
            [
                'label' => esc_html__('Space Bottom', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'post-style' => 's2',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'meta_typography',
                'exclude' => [
                    'line_height',
                    'letter_spacing',
                    'font_size',
                ],
                'selector' => '
				{{WRAPPER}} .chk-post-item__cat,
				{{WRAPPER}} .chk-post-item__meta li',
            ]
        );

        $this->add_responsive_control(
            'meta_post_alignment',
            [
                'label' => esc_html__('Meta alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'chakavak'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'chakavak'),
                        'icon' => 'eicon-h-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'chakavak'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'selectors_dictionary' => [
                    'left' => 'flex-end',
                    'right' => 'flex-start',
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__meta' => 'justify-content: {{VALUE}};',
                ],
                'condition' => [
                    'post_style!' => 's1'
                ]
            ]
        );

        $this->add_control(
            'post_meta_date',
            [
                'label' => esc_html__('Post Date', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'show_date' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'meta_date_color',
            [
                'label' => __('Date color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__date' => 'color:{{VALUE}}',
                ],
                'condition' => [
                    'show_date' => 'yes',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'meta_date_bgcolor',
                'label' => esc_html__('Background date', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .chk-post-item__date',
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Background date', 'chakavak'),
                        'default' => 'classic',
                    ],
                ],
                'condition' => [
                    'show_date' => 'yes',
                    'post_style' => 's1',
                ]
            ]
        );

        $this->add_control(
            'meta_cat_heading',
            [
                'label' => __('Category', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'show_cat' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'meta_cat_color',
            [
                'label' => __('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__cat' => 'color:{{VALUE}}',
                ],
                'condition' => [
                    'show_cat' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'meta_cat_hover_color',
            [
                'label' => __('Hover Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__cat a:is(:hover,:focus)' => 'color:{{VALUE}}',
                ],
                'condition' => [
                    'show_cat' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'meta_author_heading',
            [
                'label' => __('Author', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'show_author' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'meta_author_color',
            [
                'label' => __('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__author' => 'color:{{VALUE}}',
                ],
                'condition' => [
                    'show_author' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'meta_comments_heading',
            [
                'label' => __('Comments', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'show_comments' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'meta_comments_color',
            [
                'label' => __('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__comment' => 'color:{{VALUE}}',
                ],
                'condition' => [
                    'show_comments' => 'yes',
                ]
            ]
        );

        $this->end_controls_section();

        /* button style */
        $this->start_controls_section(
            'button_styling',
            [
                'label' => esc_html__('Button Readmore', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'exclude' => [
                    'letter_spacing',
                ],
                'selector' => '{{WRAPPER}} .chk-post-item__readmore a',
                'condition' => [
                    'post-style!' => 's1',
                ]
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'post-style!' => 's1',
                ]
            ]
        );

        $this->add_responsive_control(
            'button_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_bg_overlay',
            [
                'label' => __('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item.s2 .chk-post-item__thumbnail:before' => 'background-color:{{VALUE}}',
                ],
                'condition' => [
                    'post-style' => 's2',
                ]
            ]
        );

        $this->start_controls_tabs('button_tabs');

        $this->start_controls_tab(
            'button_normal',
            [
                'label' => esc_html__('Normal', 'chakavak')
            ]
        );

        $this->add_control(
            'button_bg_normal',
            [
                'label' => __('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a' => 'background-color:{{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'button_color_normal',
            [
                'label' => __('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a' => 'color:{{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'button_border_color_normal',
            [
                'label' => __('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a' => 'border-color:{{VALUE}}',
                ]
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover',
            [
                'label' => esc_html__('hover', 'chakavak')
            ]
        );

        $this->add_control(
            'button_bg_hover',
            [
                'label' => __('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a:is(:hover,:focus)' => 'background-color:{{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label' => __('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a:is(:hover,:focus)' => 'color:{{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label' => __('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-post-item__readmore a:is(:hover,:focus)' => 'border-color:{{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        Chakavak_Controls_Elementor::style_nav_carousel($this, ['ca_navigation' => 'yes']);
        Chakavak_Controls_Elementor::style_dots_carousel($this, ['ca_pagination' => 'yes']);
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $data_rtl = is_rtl() ? 'rtl' : 'ltr';
        $classes = [];
        $classes[] = 'chk-posts-group chk-posts-carousel swiper-slider-wrapper';
        $this->add_render_attribute('wrapper', 'class', $classes);

        //post item-class
        $post_cls = [
            'chk-post-item',
            $settings['post_style'],
            'pos-relative z-1 w-full overflow-hidden transition-all z-1',
        ];
        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div <?php echo $this->get_render_attribute_string('wrapper'); ?> dir="<?php echo $data_rtl; ?>">
                <div class="swiper-content-inner">
                    <div class="init-carousel-swiper swiper" data-carousel="<?php echo Chakavak_Controls_Elementor::get_carousel_settings($this); ?>">
                        <div class="swiper-wrapper">
                            <?php
                            $perpage = intval(intval($settings['posts_per_page']) ?? 9);
                            $paged = get_query_var('paged') ?: 1;
                            $args = [
                                'post_type' => 'post',
                                'post_status' => 'publish',
                                'order' => $settings['order'],
                                'posts_per_page' => $perpage,
                                'paged' => $paged,
                            ];

                            if (!empty($settings['post_cats'])) {
                                $args['tax_query'] = [
                                    [
                                        'taxonomy' => 'category',
                                        'field' => 'term_id',
                                        'terms' => $settings['post_cats'],
                                        'operator' => 'IN',
                                    ],
                                ];
                            }

                            if (!empty($settings['post_ids'])) {
                                $args['post__in'] = $settings['post_ids'];
                            }

                            $query = new WP_Query($args);
                            while ($query->have_posts()) : $query->the_post();
                                $post_id = get_the_ID();
                                $title = get_the_title($post_id);
                                $permalink = get_permalink($post_id);

                                //Get thumbnail image
                                $thumbnail_id = get_post_thumbnail_id($post_id);
                                $imgUrl = $thumbnail_id ? wp_get_attachment_url($thumbnail_id) : '';

                                if ($thumbnail_id) {
                                    $image_size = $settings['image_size'] === 'full' ? 'full' : Group_Control_Image_Size::get_attachment_image_src($thumbnail_id, 'image', $settings);
                                    $image_html = '<img src="' . esc_url($image_size ? $image_size : $imgUrl) . '" alt="' . esc_attr($title) . '" />';
                                } else {
                                    $image_html = '';
                                }

                                //get category
                                $terms = get_the_terms($post_id, 'category');
                                $categories = !empty($terms) ? implode(", ", wp_list_pluck($terms, 'name')) : '';

                                //get id posts
                                $cate_id = '';
                                if ($terms) {
                                    foreach ($terms as $term) {
                                        $cate_id .= 'category-' . $term->term_id . ' ';
                                    }
                                }
                                echo '<div class="swiper-slide ' . esc_attr($cate_id) . '">';
                                include CHAKAVAK_ELEMENTOR_DIR . 'widgets/layouts/posts/item-' . $settings['post_style'] . '.php';
                                echo '</div>';
                            endwhile;
                            wp_reset_postdata();
                            ?>
                        </div>
                    </div>
                </div>
                <?php echo($settings['ca_pagination'] ? '<div class="swiper-pagination"></div>' : ''); ?>
                <?php echo($settings['ca_navigation'] ? '<div class="swiper-nav-next"></div><div class="swiper-nav-prev"></div>' : ''); ?>
            </div>
        </div>
        <?php
    }

    protected function content_template()
    {
    }
}


Plugin::instance()->widgets_manager->register(new chakavak_Posts_Carousel());
