<?php

namespace Elementor;

use Chakavak_Controls_Elementor;
use WP_Query;

defined('ABSPATH') or exit;

/**
 * Widget Name: Portfolio Grid
 */
class Chakavak_Portfolio_Grid extends Widget_Base
{
    const NAME = 'chk_portfolio_grid';
    const CATEGORY = 'chakavak_widgets';

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_name()
    {
        return self::NAME;
    }

    public function get_title()
    {
        return esc_html__('Portfolio Grid', 'chakavak');
    }

    public function get_keywords()
    {
        return ['portfolio', 'grid', 'project', esc_html__('portfolio', 'chakavak'), esc_html__('projects', 'chakavak')];
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-gallery-grid';
    }

    public function get_style_depends()
    {
        return ['chk-portfolio'];
    }

    public function get_script_depends()
    {
        return ['chk-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'section_content',
            [
                'label' => esc_html__('Content', 'chakavak'),
            ]
        );


        $this->add_control(
            'filter',
            [
                'label' => __('Show Filter', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'chakavak'),
                'label_off' => __('Hide', 'chakavak'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'all_text',
            [
                'label' => __('All Text', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'default' => __('All', 'chakavak'),
                'condition' => [
                    'filter' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'pagination',
            [
                'label' => esc_html__('Pagination', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'label_off' => esc_html__('No', 'chakavak'),
                'label_on' => esc_html__('Yes', 'chakavak'),
            ]
        );

        $this->add_control(
            'portfolio_style',
            [
                'label' => esc_html__('Display Style', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 's1',
                'options' => [
                    's1' => esc_html__('Style 1', 'chakavak'),
                    's2' => esc_html__('Style 2', 'chakavak'),
                ]
            ]
        );

        $this->add_control(
            'show_cat',
            [
                'label' => __('Show Category', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'true',
                'options' => chakavak_true_false(),
            ]
        );

        $this->add_control(
            'title_tag',
            [
                'label' => esc_html__('Title Tag', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => chakavak_heading(),
                'default' => 'h3',
                'condition' => [],
                'separator' => 'none',
            ]
        );

        $this->add_control(
            'query_heading',
            [
                'label' => esc_html__('Query', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'category_ids',
            [
                'label' => __('Select Categories', 'chakavak'),
                'type' => Controls_Manager::SELECT2,
                'options' => chakavak_get_categories_list('portfolio_cat'),
                'multiple' => true,
                'label_block' => true,
                'placeholder' => __('All Categories', 'chakavak'),
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label' => __('Show Number portfolio', 'chakavak'),
                'type' => Controls_Manager::NUMBER,
                'default' => '9',
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label' => esc_html__('Order by', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'date',
                'options' => [
                    'date' => esc_html__('Date', 'chakavak'),
                    'title' => esc_html__('Title', 'chakavak'),
                    'rand' => esc_html__('Random', 'chakavak'),
                    'menu_order' => esc_html__('Menu Order', 'chakavak'),
                ]
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'DESC',
                'options' => [
                    'DESC' => esc_html__('Descending', 'chakavak'),
                    'ASC' => esc_html__('Ascending', 'chakavak'),
                ]
            ]
        );

        $this->end_controls_section();

        Chakavak_Controls_Elementor::add_control_grid($this);

        Chakavak_Controls_Elementor::filter_style($this, ['filter' => 'yes']);
        Chakavak_Controls_Elementor::pagination_style($this, ['pagination' => 'yes']);

        //Item
        $this->start_controls_section(
            'item',
            [
                'label' => esc_html__('Item', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item.s2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'portfolio_style' => 's2',
                ]
            ]
        );

        $this->add_responsive_control(
            'item_border_radius',
            [
                'label' => esc_html__('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'separator' => 'before',
                'exclude' => ['color'],
                'selector' => '{{WRAPPER}} .chk-portfolio-item',
            ]
        );

        //tabs item
        $this->start_controls_tabs('tabs_item');

        $this->start_controls_tab(
            'tab_item_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'item_border_color_normal',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'item_bgcolor_normal',
                'label' => esc_html__('Background Color', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .chk-portfolio-item.s2',
                'condition' => [
                    'portfolio_style' => 's2',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_shadow_normal',
                'separator' => 'before',
                'selector' => '{{WRAPPER}} .chk-portfolio-item',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_item_hover',
            [
                'label' => __('Hover', 'chakavak'),
            ]
        );

        $this->add_control(
            'item_border_color_hover',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'content_box_shadow_hover',
                'selector' => '{{WRAPPER}} .chk-portfolio-item:hover',
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        //image
        $this->start_controls_section(
            'portfolio_image_style',
            [
                'label' => esc_html__('Thumbnail', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'image',
                'label' => esc_html__('Image size', 'chakavak'),
                'default' => 'chk_portfolio',
            ]
        );

        $this->add_responsive_control(
            'thumbnail_border_radius',
            [
                'label' => esc_html__('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .s2 .chk-portfolio-item__thumbnail' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'portfolio_style' => 's2',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label' => esc_html__('Thumbnail shadow', 'chakavak'),
                'name' => 'thumbnail_shadow',
                'fields_options' => [
                    'box_shadow_type' =>
                        [
                            'default' => 'yes'
                        ],
                    'box_shadow' => [
                        'default' =>
                            [
                                'horizontal' => 0,
                                'vertical' => 5,
                                'blur' => 10,
                                'spread' => 0,
                                'color' => 'rgba(0,0,0,0.06)'
                            ]
                    ]
                ],
                'selector' => '{{WRAPPER}} .chk-portfolio-item__thumbnail',
                'condition' => [
                    'portfolio_style' => 's2',
                ]
            ]
        );

        $this->end_controls_section();

        //content
        $this->start_controls_section(
            'content_style',
            [
                'label' => esc_html__('Content', 'chakavak'),
                'tab' => controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'content_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'content_bgcolor',
                'label' => esc_html__('Background Color', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .s1 chk-portfolio-item__content',
                'condition' => [
                    'portfolio_style' => 's1',
                ]
            ]
        );

        $this->add_responsive_control(
            'content_border_radius',
            [
                'label' => esc_html__('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .s1 chk-portfolio-item__content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'portfolio_style' => 's1',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label' => esc_html__('Content box shadow', 'chakavak'),
                'name' => 'content_shadow',
                'selector' => '{{WRAPPER}} .s1 chk-portfolio-item__content',
                'condition' => [
                    'portfolio_style' => 's1',
                ]
            ]
        );

        //category
        $this->add_control(
            'cat_heading',
            [
                'label' => esc_html__('Category', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_cat' => 'true',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cat_typography',
                'selector' => '{{WRAPPER}} .chk-portfolio-item__cat',
                'condition' => [
                    'show_cat' => 'true',
                ]
            ]
        );

        $this->add_control(
            'cat_color',
            [
                'label' => esc_html__('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__cat' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_cat' => 'true',
                ]
            ]
        );

        $this->add_control(
            'cat_color_hover',
            [
                'label' => esc_html__('Text Color hover', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .s2:hover .chk-portfolio-item__cat' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_cat' => 'true',
                    'portfolio_style' => 's2',
                ]
            ]
        );

        //title
        $this->add_control(
            'title_heading',
            [
                'label' => esc_html__('Title', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .chk-portfolio__title',
            ]
        );

        $this->add_control(
            'title_inline',
            [
                'label' => esc_html__('Title inline', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_on' => esc_html__('Yes', 'chakavak'),
                'label_off' => esc_html__('No', 'chakavak'),
                'return_value' => 'no',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_color_hover',
            [
                'label' => esc_html__('Title color hover', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .s1 .chk-portfolio-item__title a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .s2:hover .chk-portfolio-item__title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        //category
        $this->add_control(
            'button_heading',
            [
                'label' => esc_html__('Button', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'button_radius',
            [
                'label' => __('Border Radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'label' => esc_html__('Border', 'chakavak'),
                'selector' => '{{WRAPPER}} .chk-portfolio-item__readmore>a ',
                'exclude' => ['color'],
                'separator' => 'before',
            ]
        );


        $this->start_controls_tabs('button_tabs');

        $this->start_controls_tab(
            'button_tab_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_color_normal',
            [
                'label' => esc_html__('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_bg_normal',
            [
                'label' => esc_html__('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_border_color_normal',
            [
                'label' => esc_html__('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a' => 'border-color: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'or', // or
                    'terms' => [
                        [
                            'name' => 'button_border_width[top]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                        [
                            'name' => 'button_border_width[right]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                        [
                            'name' => 'button_border_width[left]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                        [
                            'name' => 'button_border_width[bottom]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                    ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_tab_hover',
            [
                'label' => __('Hover', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label' => esc_html__('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_bg_hover',
            [
                'label' => esc_html__('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label' => esc_html__('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-portfolio-item__readmore>a:hover' => 'border-color: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'or', // or
                    'terms' => [
                        [
                            'name' => 'button_border_width[top]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                        [
                            'name' => 'button_border_width[right]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                        [
                            'name' => 'button_border_width[left]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                        [
                            'name' => 'button_border_width[bottom]',
                            'operator' => '!=',
                            'value' => ['0'],
                        ],
                    ],
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    public static function get_query_args($settings)
    {
        global $paged;
        if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        } else if (get_query_var('page')) {
            $paged = get_query_var('page');
        } else {
            $paged = 1;
        }
        $query_args = [
            'post_type' => 'portfolio',
            'paged' => $paged,
            'posts_per_page' => (!empty($settings['posts_per_page'])) ? $settings['posts_per_page'] : 6,
            'orderby' => (!empty($settings['orderby'])) ? $settings['orderby'] : 'date',
            'order' => (!empty($settings['order'])) ? $settings['order'] : 'DESC',
            'ignore_sticky_posts' => 1,
            'post_status' => 'publish',
        ];

        // Handle category selection
        if (!empty($settings['category_ids'])) {
            $query_args['tax_query'] = [
                [
                    'taxonomy' => 'portfolio_cat',
                    'field' => 'term_id',
                    'terms' => $settings['category_ids'],
                    'operator' => 'IN',
                ]
            ];
        }

        // Handle menu order if selected
        if ($settings['orderby'] === 'menu_order') {
            $query_args['orderby'] = 'menu_order title';
        }

        return $query_args;
    }

    public function query_posts()
    {
        $query_args = $this->get_query_args($this->get_settings());
        return new WP_Query($query_args);
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $data_rtl = is_rtl() ? 'true' : 'false';
        $query = $this->query_posts();
        if (!$query->have_posts()) {
            return;
        }
        $classes = [
            'chk-portfolio-group',
            'chk-layout-grid',
            'chk-portfolio-grid',
            esc_attr('chk-portfolio-grid__' . $this->get_id()),
        ];
        $this->add_render_attribute('wrapper', 'class', array_filter($classes));
        Chakavak_Controls_Elementor::get_grid_settings($this, 'chk-portfolio-grid__items');

        // Portfolio item attributes
        $this->add_render_attribute('portfolio-item', 'class', [
            'chk-portfolio-item',
            $settings['portfolio_style'],
            'pos-relative',
            'z-1',
            'w-full',
            'transition-all',
            'overflow-hidden',
            'chk-bg-before'
        ]);

        // Content attributes
        $this->add_render_attribute('portfolio-content', 'class', [
            'chk-portfolio-item__content',
            'pos-relative',
        ]);
        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div <?php echo $this->get_render_attribute_string('wrapper'); ?> data-rtl="<?php echo $data_rtl; ?>">

                <?php if ('yes' === $settings['filter']) { ?>
                    <div class="chk-filter__items">

                        <?php if ($settings['all_text']) { ?>
                            <a class="chk-filter__item-link is-checked" href="javascript:void(0)" data-filter="*">
                                <?php echo esc_html($settings['all_text']); ?>
                            </a>
                        <?php } ?>

                        <?php
                        if (!empty($settings['category_ids'])) {
                            foreach ($settings['category_ids'] as $term_id) {
                                $term = get_term($term_id, 'portfolio_cat');
                                if (!is_wp_error($term) && $term) {
                                    $cat_name = $term->name;
                                    $cat_id = 'ca-' . $term->term_id;
                                    ?>
                                    <a class="chk-filter__item-link" href='javascript:void(0)' data-filter='.<?php echo esc_attr($cat_id); ?>'>
                                        <?php echo esc_html($cat_name); ?>
                                    </a>
                                    <?php
                                }
                            }
                        } else {
                            $categories = get_terms([
                                'taxonomy' => 'portfolio_cat',
                                'hide_empty' => true
                            ]);

                            foreach ($categories as $category) {
                                $cat_name = $category->name;
                                $cat_id = 'cat-' . $category->term_id;
                                ?>
                                <a class="chk-filter__item-link" href='javascript:void(0)' data-filter='.<?php echo esc_attr($cat_id); ?>'>
                                    <?php echo esc_html($cat_name); ?>
                                </a>
                                <?php
                            }
                        } ?>
                    </div>
                <?php } ?>

                <div <?php echo $this->get_render_attribute_string('grid') ?>>
                    <?php
                    while ($query->have_posts()) :
                        $query->the_post();
                        $post_id = get_the_ID();
                        $title = get_the_title($post_id);
                        $thumbnail_id = get_post_thumbnail_id($post_id);
                        $permalink = get_permalink($post_id);
                        // Generate image HTML
                        $image_html = '';
                        if ($thumbnail_id) {
                            if ($settings['image_size'] === 'full') {
                                $image_html = wp_get_attachment_image($thumbnail_id, 'full');
                            } else {
                                $image_url = Group_Control_Image_Size::get_attachment_image_src($thumbnail_id, 'image', $settings);
                                if (!$image_url) {
                                    $image_url = wp_get_attachment_url($thumbnail_id);
                                }

                                $image_srcset = wp_get_attachment_image_srcset($thumbnail_id, $settings['image_size']);
                                $image_sizes = wp_get_attachment_image_sizes($thumbnail_id, $settings['image_size']);

                                $image_alt = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);
                                if (empty($image_alt)) {
                                    $image_alt = esc_attr($title);
                                }

                                $image_html = sprintf(
                                    '<img src="%s" alt="%s" srcset="%s" sizes="%s" />',
                                    esc_url($image_url),
                                    esc_attr($image_alt),
                                    esc_attr($image_srcset),
                                    esc_attr($image_sizes)
                                );
                            }
                        }

                        // Get categories
                        $terms = get_the_terms($post_id, 'portfolio_cat');
                        $categories = [];
                        if ($terms && !is_wp_error($terms)) {
                            $categories = wp_list_pluck($terms, 'name');
                        }
                        $categories_string = implode(', ', $categories);

                        $cate_id = '';
                        foreach ((array)$terms as $cate) {
                            if ($terms) {
                                $cate_id .= 'cat-' . $cate->term_id . ' ';
                            }
                        }

                        echo '<div class="chk-prtfolio-item__wrap col-xs ' . esc_attr($cate_id) . '">';
                        include CHAKAVAK_ELEMENTOR_DIR . 'widgets/layouts/portfolio/item-' . $settings['portfolio_style'] . '.php';
                        echo '</div>';
                    endwhile;
                    wp_reset_postdata(); ?>
                </div>
            </div>

            <?php if ('yes' === $settings['pagination']) :
                echo chakavak_standard_pagination($query);
            endif; ?>

        </div>
        <?php
        if (Plugin::instance()->editor->is_edit_mode()) {
            $this->render_editor_script();
        }
    }

    protected function render_editor_script()
    {
        ?>
        <script type="text/javascript">
            jQuery(document).ready(function ($) {
                var wrapper = '.chk-portfolio-grid__' + '<?php echo $this->get_id(); ?>',
                    $this = $(wrapper),
                    $isoGrid = $this.children('.chk-portfolio-grid__items'),
                    $btns = $this.children('.chk-filter__items'),
                    is_rtl = $this.data('rtl');
                $this.imagesLoaded(function () {
                    var $grid = $isoGrid.isotope({
                        itemSelector: '.chk-prtfolio-item__wrap',
                        percentPosition: true,
                        originLeft: is_rtl,
                        masonry: {
                            columnWidth: '.chk-prtfolio-item__wrap',
                        }
                    });

                    $btns.on('click', '.chk-filter__item-link', function () {
                        var filterValue = $(this).attr('data-filter');
                        $grid.isotope({
                            filter: filterValue,
                            originLeft: is_rtl
                        });
                    });

                    $this.find('.chk-prtfolio-item__wrap').resize(function () {
                        $grid.isotope('layout');
                    });

                    $btns.each(function (i, btns) {
                        var btns = $(btns);

                        btns.on('click', '.chk-filter__item-link', function () {
                            btns.find('.is-checked').removeClass('is-checked');
                            $(this).addClass('is-checked');
                        });
                    });
                });
            })
        </script>
        <?php
    }

    protected function content_template()
    {
    }
}

Plugin::instance()->widgets_manager->register(new Chakavak_Portfolio_Grid());
