<?php

namespace Elementor;

use Chakavak_Controls_Elementor;

defined('ABSPATH') or exit;

/**
 * Widget Name: Message box
 */
class Chakavak_MessageBox extends Widget_Base
{
    const NAME = 'chk_messagebox';
    const CATEGORY = 'chakavak_widgets';

    public function get_name()
    {
        return self::NAME;
    }

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_title()
    {
        return __('Message box', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-alert';
    }

    public function get_keywords()
    {
        return ['message', 'alert', 'notifaction', esc_html__('Massage box', 'chakavak')];
    }

    public function get_style_depends()
    {
        return ['chk-elementor-widgets'];
    }

    public function get_script_depends()
    {
        return ['chk-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'contact_settings',
            [
                'label' => esc_html__('Message Box', 'chakavak'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'message_title',
            [
                'label' => esc_html__('Title', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'placeholder' => __('Enter title.', 'chakavak'),
            ]
        );

        $this->add_control(
            'message_desc',
            [
                'label' => esc_html__('Description', 'chakavak'),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 4,
                'default' => __('We help ambitious businesses like yours generate.', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_label',
            [
                'label' => __('Button Label', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'label_block' => true,
                'default' => __('Click here', 'chakavak'),
                'placeholder' => __('Enter button label', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_link',
            [
                'label' => __('Button Link', 'chakavak'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => 'https://your-link.com',
                'default' => [
                    'url' => '#',
                ],
                'condition' => [
                    'button_label!' => ''
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'message_style',
            [
                'label' => __('Message Style', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'message_content_gap',
            [
                'label' => __('Space between', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__wrap' => 'gap: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'message_wrap_alignment!' => 'space-between',
                ],
            ]
        );

        $this->add_responsive_control(
            'message_padding',
            [
                'label' => __('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => '5',
                    'bottom' => '5',
                    'right' => '5',
                    'left' => '5',
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'message_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'message_background',
                'label' => esc_html__('Background', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Background color', 'chakavak'),
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => 'var(--chk-c-primary)',
                    ],
                ],
                'selector' => '{{WRAPPER}} .chk-message-box',

            ]
        );

        //desc
        $this->add_control(
            'desc_style',
            [
                'label' => __('Description', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'message_desc!' => '',
                ],
            ]
        );

        $this->add_control(
            'desc_tag',
            [
                'label' => esc_html__('Description Tag', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => chakavak_heading(),
                'default' => 'p',
                'condition' => [
                    'message_desc!' => '',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'desc_typography',
                'condition' => [
                    'message_desc!' => '',
                ],
                'selector' => '{{WRAPPER}} .chk-message-box__desc',
            ]
        );

        $this->add_control(
            'desc_color',
            [
                'label' => esc_html__('Description color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__desc' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'message_desc!' => '',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'desc_text_shadow',
                'fields_options' => [
                    'box_shadow_type' =>
                        [
                            'default' => 'yes'
                        ],
                    'box_shadow' => [
                        'default' =>
                            [
                                'horizontal' => 0,
                                'vertical' => 3,
                                'blur' => 5,
                                'spread' => 0,
                                'color' => 'rgba(0,0,0,0.1)'
                            ]
                    ]
                ],
                'selector' => '{{WRAPPER}} .chk-message-box__desc',
                'condition' => [
                    'message_desc!' => '',
                ],
            ]
        );

        //close
        $this->add_control(
            'close_heading',
            [
                'label' => esc_html__('Close button', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_responsive_control(
            'close_size',
            [
                'label' => esc_html__('Icon size', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__close' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'close_color',
            [
                'label' => __('Text color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__close' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'button_style',
            [
                'label' => __('Button style', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'button_label!' => '',
                    'button_link[url]!' => ''
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'selector' => '{{WRAPPER}} .chk-message-box__btn',
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => __('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 30,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_border_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 9999,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'color' => [
                        'default' => 'rgba(255,255,255,.3)',
                    ],
                    'width' => [
                        'default' => [
                            'top' => '1',
                            'right' => '1',
                            'bottom' => '1',
                            'left' => '1',
                            'isLinked' => true,
                        ]
                    ],
                ],
                'selector' => '{{WRAPPER}} .chk-message-box__btn',
            ]
        );

        $this->start_controls_tabs('tabs_button_style');

        $this->start_controls_tab(
            'tab_button_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );
        $this->add_control(
            'button_text_color',
            [
                'label' => __('Text color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'button_bg_color',
            [
                'label' => __('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn' => 'background-color:{{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow',
                'selector' => ' {{WRAPPER}} .chk-message-box__btn',
            ]
        );
        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_button_hover',
            [
                'label' => __('Hover', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label' => __('Text color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn:is(:hover,:focus)' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_bg_color_hover',
            [
                'label' => __('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn:is(:hover,:focus)' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label' => __('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-message-box__btn:is(:hover,:focus)' => 'border-color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow_hover',
                'selector' => ' {{WRAPPER}} .chk-message-box__btn:is(:hover,:focus)',
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div class="chk-message-box pos-relative">
                <div class="chk-message-box__wrap d-flex w-full pos-relative">
                    <?php if (!empty($settings['message_desc'])) {
                        $tag_desc = $settings['desc_tag'];
                        $cls_desc = "chk-message-box__desc";
                        echo "<{$tag_desc} class=\"{$cls_desc}\">";
                        echo wp_kses_post($settings['message_desc']);
                        echo "</{$tag_desc}>";
                    } ?>

                    <?php if (!empty($settings['button_link']['url'])) {
                        echo '<div class="chk-message-box__button">';
                        $cls_btn = "chk-message-box__btn if-center transition-all";
                        Chakavak_Controls_Elementor::render_start_link($this, $settings['button_link'], $cls_btn);
                        if (!empty($settings['button_label'])) {
                            echo '<span>' . esc_html($settings['button_label']) . '</span>';
                        } else {
                            echo '<span>' . esc_html__('Click here', 'chakavak') . '</span>';
                        }
                        Chakavak_Controls_Elementor::render_end_link($this, $settings['button_link']);
                        echo '</div>';
                    } ?>

                    <div class="chk-message-box__close transition-all cursor-pointer">
                        <i class="ti-close"></i>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register(new Chakavak_MessageBox());
