<?php

namespace Elementor;

use Chakavak_Controls_Elementor;

defined('ABSPATH') or exit;

/**
 * Widget Name: Image Slider
 */
class Chakavak_Image_Slider extends Widget_Base
{
    const NAME = 'chk_image_slider';
    const CATEGORY = 'chakavak_widgets';

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_name()
    {
        return self::NAME;
    }

    public function get_title()
    {
        return esc_html__('Image Slider', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-slider-push';
    }

    public function get_keywords()
    {
        return ['image', 'slider', esc_html__('carousel', 'chakavak'), esc_html__('image slider', 'chakavak')];
    }

    public function get_style_depends()
    {
        return [
            'swiper-bundle',
            'chk-image-slider'
        ];
    }

    public function get_script_depends()
    {
        return [
            'swiper-bundle',
            'chk-widgets'
        ];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'image_content',
            [
                'label' => esc_html__('Image', 'chakavak'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $repeater = new Repeater();
        $repeater->add_control(
            'title',
            [
                'label' => __('Name', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Title', 'chakavak'),
            ]
        );

        $repeater->add_control(
            'image_item',
            [
                'label' => __('Image', 'chakavak'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ]
            ]
        );

        $repeater->add_control(
            'link',
            [
                'label' => __('Link', 'chakavak'),
                'type' => Controls_Manager::URL,
            ]
        );
        $this->add_control(
            'images_slider',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'show_label' => false,
                'default' => [
                    [
                        'title' => __('Image 1', 'chakavak'),
                        'image_item' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'title' => __('Image 2', 'chakavak'),
                        'image_item' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'title' => __('Image 3', 'chakavak'),
                        'image_item' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ]
                ],
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{title}}}',
            ]
        );

        $this->add_control(
            'image_popup',
            [
                'label' => __('Enable LightBox', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
            ]
        );

        Chakavak_Controls_Elementor::image_size($this, 'image_carousel_size', null);

        $this->end_controls_section();

        Chakavak_Controls_Elementor::add_control_carousel($this, false);

        //Item Style
        $this->start_controls_section(
            'image_style',
            [
                'label' => esc_html__('Image', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'image_vertical_align',
            [
                'label' => __('Vertical Align', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => __('Start', 'chakavak'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'center' => [
                        'title' => __('Center', 'chakavak'),
                        'icon' => 'eicon-v-align-middle',
                    ],
                    'flex-end' => [
                        'title' => __('End', 'chakavak'),
                        'icon' => 'eicon-v-align-bottom',
                    ],
                ],
                'condition' => [
                    'tshow!' => '1',
                ],
                'selectors' => [
                    '{{WRAPPER}} .swiper-wrapper' => 'align-items: {{VALUE}};',
                    '{{WRAPPER}} .swiper-wrapper >*' => 'flex-shrink: 0;',
                ],
            ]
        );

        $this->add_control(
            'image_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-image-slider__item img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_width',
            [
                'label' => __('Width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-image-slider__item img' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Css_Filter::get_type(),
            [
                'name' => 'img_css_filters',
                'selector' => '{{WRAPPER}} .chk-image-slider__item img',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'button_style',
            [
                'label' => esc_html__('Button', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label' => __('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-image-slider__link' => 'color:{{VALUE}}'
                ]
            ]
        );
        $this->add_control(
            'button_bgcolor',
            [
                'label' => __('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-image-slider__link' => 'background-color:{{VALUE}}'
                ]
            ]
        );

        $this->end_controls_section();

        Chakavak_Controls_Elementor::style_nav_carousel($this, ['ca_navigation' => 'yes']);
        Chakavak_Controls_Elementor::style_dots_carousel($this, ['ca_pagination' => 'yes']);
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $data_rtl = is_rtl() ? 'rtl' : 'ltr';
        $classes = [];
        $classes[] = 'chk-image-slider swiper-slider-wrapper';
        $this->add_render_attribute('wrapper', 'class', $classes);
        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div <?php echo $this->get_render_attribute_string('wrapper'); ?> dir="<?php echo $data_rtl; ?>">
                <div class="swiper-content-inner">
                    <div class="init-carousel-swiper swiper" data-carousel="<?php echo Chakavak_Controls_Elementor::get_carousel_settings($this); ?>">
                        <div class="swiper-wrapper">
                            <?php
                            $id_int = substr($this->get_id_int(), 0, 3);
                            foreach ($settings['images_slider'] as $key => $item):
                                $title = $item['title'];
                                $image_url = Group_Control_Image_Size::get_attachment_image_src($item['image_item']['id'], 'image_carousel_size', $settings);
                                if (!$image_url && isset($item['image_item']['url'])) {
                                    $image_url = $item['image_item']['url'];
                                }
                                $image_html = '<img src="' . esc_attr($image_url) . '" alt="' . esc_attr($title) . '">';
                                $url_page = $item['link'];
                                $image_popup = $settings['image_popup'];
                                $link = !empty($item['link']['url']) && empty($settings['image_popup']);
                                if (!empty($url_page['url'])) {
                                    $this->add_render_attribute('link' . $key, 'class', 'chk-image-slider__link if-center z-1');
                                    $this->add_link_attributes('link' . $key, $url_page);
                                }

                                if ($image_popup == 'yes') {
                                    $this->add_render_attribute('link_popup' . $key, 'class', 'chk-image-slider__link if-center z-1');
                                    $this->add_render_attribute('link_popup' . $key, 'href', $image_url);
                                }
                                ?>
                                <div class="swiper-slide">
                                    <figure class="chk-image-slider__item pos-relative">
                                        <?php echo $image_html; ?>
                                        <?php if (!empty($url_page['url']) || $image_popup == 'yes') : ?>
                                            <div class="chk-image-slider__item-buttons">
                                                <?php if (!empty($url_page['url'])) : ?>
                                                    <a <?php echo $this->get_render_attribute_string('link' . $key); ?>>
                                                        <i class="ti-link"></i>
                                                    </a>
                                                <?php endif; ?>
                                                <?php if ($image_popup == 'yes') : ?>
                                                    <a <?php echo $this->get_render_attribute_string('link_popup' . $key); ?>>
                                                        <i class="ti-full-screen"></i>
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </figure>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php echo($settings['ca_pagination'] ? '<div class="swiper-pagination"></div>' : ''); ?>
                <?php echo($settings['ca_navigation'] ? '<div class="swiper-nav-next"></div><div class="swiper-nav-prev"></div>' : ''); ?>
            </div>
        </div>
        <?php
    }

    protected function content_template()
    {
    }
}

Plugin::instance()->widgets_manager->register(new Chakavak_Image_Slider());
