<?php

namespace Elementor;

use Chakavak_Controls_Elementor;

defined('ABSPATH') or exit;

/**
 * Widget Name: Iconic List
 */
class Chakavak_Iconic_list extends Widget_Base
{
    const NAME = 'chk_iconic_list';
    const CATEGORY = 'chakavak_widgets';

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_name()
    {
        return self::NAME;
    }

    public function get_title()
    {
        return esc_html__('Iconic list', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-bullet-list';
    }

    public function get_keywords()
    {
        return ['list', 'list icon', esc_html__('iconic list', 'chakavak')];
    }

    public function get_style_depends()
    {
        return [
            'chk-elementor-widgets'
        ];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'section_icon',
            [
                'label' => __('Icon List', 'chakavak'),
            ]
        );

        $this->add_control(
            'sw_title',
            [
                'label' => __('Show Title', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'chakavak'),
                'label_off' => __('No', 'chakavak'),
                'default' => 'no',
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'chakavak'),
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'placeholder' => esc_html__('Enter title', 'chakavak'),
                'default' => esc_html__('List menu', 'chakavak'),
                'condition' => [
                    'sw_title' => 'yes',
                ],
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'title_tag',
            [
                'label' => esc_html__('Title Tag', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => chakavak_heading(),
                'default' => 'h4',
                'condition' => [
                    'sw_title' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'list_heading',
            [
                'label' => esc_html__('List', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'view',
            [
                'label' => __('Layout', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'vertical',
                'options' => [
                    'vertical' => [
                        'title' => __('Default', 'chakavak'),
                        'icon' => 'eicon-editor-list-ul',
                    ],
                    'horizontal' => [
                        'title' => __('Inline', 'chakavak'),
                        'icon' => 'eicon-ellipsis-h',
                    ],
                ],
                'render_type' => 'template', /*Live load*/
                'separator' => 'after',
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'label',
            [
                'label' => __('Label', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'placeholder' => __('Enter label item', 'chakavak'),
                'default' => __('List menu item', 'chakavak'),
            ]
        );

        $repeater->add_control(
            'icon_type',
            [
                'label' => __('Icon Type', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'font',
                'options' => [
                    'none' => __('None', 'chakavak'),
                    'font' => __('Font Icon', 'chakavak'),
                    'text' => __('Text', 'chakavak'),
                    'image' => __('Image Icon', 'chakavak'),
                ]
            ]
        );

        $repeater->add_control(
            'icon_font',
            [
                'label' => __('Icon', 'chakavak'),
                'type' => Controls_Manager::ICONS,
                'label_block' => true,
                'default' => [
                    'value' => 'ti ti-check',
                    'library' => 'theme-icons',
                ],
                'fa4compatibility' => 'icon',
                'condition' => [
                    'icon_type' => 'font',
                ]
            ]
        );

        $repeater->add_control(
            'icon_image',
            [
                'label' => esc_html__('Image', 'chakavak'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => CHAKAVAK_ELEMENTOR_ASSETS . 'images/icon.png',
                ],
                'condition' => [
                    'icon_type' => 'image',
                ]
            ]
        );

        $repeater->add_control(
            'icon_text',
            [
                'label' => __('Text', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'placeholder' => __('Enter text icon', 'chakavak'),
                'default' => __('01.', 'chakavak'),
                'condition' => [
                    'icon_type' => 'text',
                ]
            ]
        );

        $repeater->add_control(
            'link',
            [
                'label' => __('Link', 'chakavak'),
                'type' => Controls_Manager::URL,
                'label_block' => true,
                'placeholder' => 'https://your-link.com',
            ]
        );

        $repeater->add_responsive_control(
            'icon_vertical_alignment',
            [
                'label' => __('Icon vertical align', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__('Top', 'chakavak'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'chakavak'),
                        'icon' => 'eicon-v-align-middle',
                    ],
                    'flex-end' => [
                        'title' => esc_html__('Bottom', 'chakavak'),
                        'icon' => 'eicon-v-align-bottom',
                    ],
                ],
                'condition' => [
                    'icon_type!' => 'none',
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}' => 'align-items: {{VALUE}};',
                ],
            ]
        );


        $this->add_control(
            'list_items',
            [
                'label' => __('list items', 'chakavak'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ label }}}',
                'default' => [
                    [
                        'label' => __('Menu Item #1', 'chakavak'),
                    ],
                    [
                        'label' => __('Menu Item #2', 'chakavak'),
                    ],
                    [
                        'label' => __('Menu Item #3', 'chakavak'),
                    ],
                ],
            ]
        );

        $this->add_control(
            'icon_view',
            [
                'label' => __('View icon', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'default' => __('Default', 'chakavak'),
                    'stacked' => __('Stacked', 'chakavak'),
                ],
                'default' => 'default',
                'prefix_class' => 'chk-iconic-list__',
            ]
        );

        $this->end_controls_section();

        //title style
        $this->start_controls_section(
            'title_tyle',
            [
                'label' => __('Title', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'sw_title' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .chk-iconic-list__title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('Title color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_align',
            [
                'label' => __('Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => __('Right', 'chakavak'),
                        'icon' => 'eicon-text-align-right',
                    ],
                    'center' => [
                        'title' => __('Center', 'chakavak'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'flex-end' => [
                        'title' => __('Left', 'chakavak'),
                        'icon' => 'eicon-text-align-left',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__title' => 'text-align: {{VALUE}};',
                ]
            ]
        );

        $this->add_responsive_control(
            'tite_margin_bottom',
            [
                'label' => __('Spacing bottom', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 150,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 1,
                        'max' => 30,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__heading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        //list items style
        $this->start_controls_section(
            'section_menu_list',
            [
                'label' => __('List', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'space_between',
            [
                'label' => __('Space Between', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__vertical > li:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .chk-iconic-list__horizontal' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'list_menu_align',
            [
                'label' => __('Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => __('Right', 'chakavak'),
                        'icon' => 'eicon-text-align-right',
                    ],
                    'center' => [
                        'title' => __('Center', 'chakavak'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'flex-end' => [
                        'title' => __('left', 'chakavak'),
                        'icon' => 'eicon-text-align-left',
                    ],

                    'space-between' => [
                        'title' => esc_html__('Justified', 'chakavak'),
                        'icon' => 'eicon-text-align-justify',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__horizontal' => 'justify-content: {{VALUE}};',
                    '{{WRAPPER}} .chk-iconic-list__vertical' => 'align-items: {{VALUE}}',
                ]
            ]
        );

        $this->end_controls_section();

        //icon style
        $this->start_controls_section(
            'section_icon_style',
            [
                'label' => __('Icon / Number ', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'icon_typography',
                'selector' => '{{WRAPPER}} .chk-iconic-list__icon-text',
                'description' => __('Only when the icon type is equal to Text', 'chakavak'),
                'exclude' => [
                    'font_size',
                    'line_height',
                    'letter_spacing',
                    'word_spacing',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon size', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 1,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__icon' => '--icon-size: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__icon' => '--icon-padding: {{SIZE}}{{UNIT}}',
                ],
                'default' => [
                    'unit' => 'em',
                ],
                'tablet_default' => [
                    'unit' => 'em',
                ],
                'mobile_default' => [
                    'unit' => 'em',
                ],
                'range' => [
                    'px' => [
                        'max' => 20,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                ],

                'condition' => [
                    'icon_view' => 'stacked',
                ],
            ]
        );


        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => __('Space between ', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 1,
                        'max' => 20,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__item' => 'column-gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 200,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'icon_view' => 'stacked',
                ],
            ]
        );

        $this->start_controls_tabs('icon_tab_style');

        $this->start_controls_tab(
            'icon_tab_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'icon_color_normal',
            [
                'label' => __('Icon color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__icon b' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .chk-iconic-list__icon svg' => 'fill: {{VALUE}};',
                    '{{WRAPPER}} .chk-iconic-list__icon i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'icon_bgcolor_normal',
                'label' => __('Icon background color', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .chk-iconic-list__icon',
                'condition' => [
                    'icon_view' => 'stacked'
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label' => esc_html__('Icon shadow', 'chakavak'),
                'name' => 'icon_shadow_normal',
                'selector' => '{{WRAPPER}} .chk-iconic-list__icon',
                'condition' => [
                    'icon_view' => 'stacked',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_tab_hover',
            [
                'label' => __('Hover', 'chakavak'),
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label' => __('Icon color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__item:hover .chk-iconic-list__icon i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .chk-iconic-list__item:hover .chk-iconic-list__icon svg' => 'fill: {{VALUE}};',
                    '{{WRAPPER}} .chk-iconic-list__item:hover .chk-iconic-list__icon b' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'icon_bgcolor_hover',
                'label' => __('Icon background color', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .chk-iconic-list__icon:before',
                'condition' => [
                    'icon_view' => 'stacked',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label' => esc_html__('Icon shadow', 'chakavak'),
                'name' => 'icon_shadow_hover',
                'selector' => '{{WRAPPER}} .chk-iconic-list__item:hover .chk-iconic-list__icon',
                'condition' => [
                    'icon_view' => 'stacked',
                ]
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

        $this->start_controls_section(
            'label_style',
            [
                'label' => __('Label', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'item_typography',
                'selector' => '{{WRAPPER}} .chk-iconic-list__label',
            ]
        );


        $this->add_control(
            'label_color',
            [
                'label' => __('Text color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__label' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'text_color_hover',
            [
                'label' => __('Color text hover', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-iconic-list__item:hover .chk-iconic-list__label' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .chk-iconic-list__item:hover .chk-iconic-list__label a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div class="chk-iconic-list">

                <?php if ($settings['sw_title'] == 'yes') : ?>
                    <div class="chk-iconic-list__heading">
                        <?php
                        $this->add_render_attribute('title', 'class', 'chk-iconic-list__title');
                        echo '<' . $settings['title_tag'] . ' ' . $this->get_render_attribute_string('title') . '>';
                        echo esc_html($settings['title']);
                        echo '</' . $settings['title_tag'] . '>';
                        ?>
                    </div>
                <?php endif; ?>

                <?php if ($settings['list_items']):
                    $this->add_render_attribute('items', 'class', 'chk-iconic-list__items');
                    $this->add_render_attribute(
                        'items',
                        'class',
                        [
                            'chk-iconic-list__items',
                            'list-none',
                            'pos-relative',
                            'z-1 p-0 m-0',
                            'chk-iconic-list__' . $settings['view']
                        ]
                    );
                    $item_class = 'chk-iconic-list__item pos-relative d-inline-flex items-center transition-all z-1';
                    ?>
                    <ul <?php echo $this->get_render_attribute_string('items'); ?>>
                        <?php
                        foreach ($settings['list_items'] as $item) {
                            echo '<li class="elementor-repeater-item-' . $item['_id'] . ' ' . $item_class . '">';
                            if ($item['icon_type'] != 'none') {
                                echo '<span class="chk-iconic-list__icon pos-relative transition-all chk-bg-before z-1 ' . ($settings['icon_view'] === 'stacked' ? 'chk-bg-before' : '') . '">';
                                if ($item['icon_type'] == 'font') {
                                    Icons_Manager::render_icon($item['icon_font'], [
                                        'aria-hidden' => 'true',
                                        'class' => 'pos-relative'
                                    ]);
                                } elseif ($item['icon_type'] == 'image') {
                                    echo '<img src="' . esc_attr($item['icon_image']['url']) . '" alt="icon-' .
                                        $item['_id'] . '" class="pos-relative z-1">';
                                } elseif ($item['icon_type'] == 'text') {
                                    echo '<b class="chk-iconic-list__icon-text pos-relative z-1 lh-1">' . esc_html($item['icon_text']) . '</b>';
                                }
                                echo '</span>';
                            }
                            echo '<span class="chk-iconic-list__label transition-all">';
                            Chakavak_Controls_Elementor::render_start_link($this, $item['link'], 'chk-iconic-list__link');
                            echo esc_html($item['label']);
                            Chakavak_Controls_Elementor::render_end_link($this, $item['link']);
                            echo '</span>';
                            echo '</li>';
                        }
                        ?>
                    </ul>
                <?php endif; ?>
            </div>
        </div>
    <?php }
}

Plugin::instance()->widgets_manager->register(new Chakavak_Iconic_list());
