<?php

namespace Elementor;

defined('ABSPATH') or exit;

/**
 * Widget Name:  Animated Text
 */
class chakavak_Animated_Text extends Widget_Base
{
    const NAME = 'chk_animated_text';
    const CATEGORY = 'chakavak_widgets';

    public function get_name()
    {
        return self::NAME;
    }

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_title()
    {
        return __('Animated Text', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-animation-text';
    }

    public function get_keywords()
    {
        return ['animated text', 'text', 'typing', 'animation', 'animated', esc_html__('animated text', 'chakavak')];
    }

    public function get_style_depends()
    {
        return ['chk-elementor-widgets'];
    }

    public function get_script_depends()
    {
        return [
            'typed',
            'chakavak-widgets',
        ];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'animated_text_content',
            [
                'label' => __('Animated Text', 'chakavak'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_responsive_control(
            'alignment',
            [
                'label' => esc_html__('Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'chakavak'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'chakavak'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'chakavak'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $repeater = new Repeater();
        $repeater->add_control(
            'animated_text',
            [
                'label' => esc_html__('Text', 'chakavak'),
                'label_block' => true,
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Fancy text', 'chakavak'),
            ]
        );

        $this->add_control(
            'animated_texts',
            [
                'label' => esc_html__('Animated Text', 'chakavak'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ animated_text }}}',
                'default' => [
                    [
                        'animated_text' => esc_html__('First text', 'chakavak'),
                    ],
                    [
                        'animated_text' => esc_html__('Second text', 'chakavak'),
                    ],
                    [
                        'animated_text' => esc_html__('Third text', 'chakavak'),
                    ],
                    [
                        'animated_text' => esc_html__('Fourth text', 'chakavak'),
                    ],
                ],
            ]
        );

        $this->add_control(
            'animated_prefix',
            [
                'label' => esc_html__('Prefix', 'chakavak'),
                'type' => Controls_Manager::TEXTAREA,
                'placeholder' => esc_html__('Enter prefix text', 'chakavak'),
                'default' => esc_html__('Prefix text', 'chakavak'),
                'rows' => '3',
            ]
        );
        $this->add_control(
            'animated_suffix',
            [
                'label' => esc_html__('Suffix', 'chakavak'),
                'type' => Controls_Manager::TEXTAREA,
                'placeholder' => esc_html__('Enter suffix text', 'chakavak'),
                'default' => esc_html__('Suffix text', 'chakavak'),
                'rows' => '3',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'animated_settings',
            [
                'label' => esc_html__('Animated Settings', 'chakavak'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'display',
            [
                'label' => esc_html__('Display', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'chk-animated-inline' => esc_html__('Inline', 'chakavak'),
                    'chk-animated-block' => esc_html__('Block', 'chakavak'),
                ],
                'default' => 'chk-animated-inline',
            ]
        );


        $this->add_control(
            'animation_type',
            [
                'label' => esc_html__('Animation Type', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'typing',
                'options' => [
                    'typing' => esc_html__('Typing', 'chakavak'),
                    'fadeIn' => esc_html__('Fade', 'chakavak'),
                    'fadeInUp' => esc_html__('Fade Up', 'chakavak'),
                    'fadeInDown' => esc_html__('Fade Down', 'chakavak'),
                    'fadeInLeft' => esc_html__('Fade Left', 'chakavak'),
                    'fadeInRight' => esc_html__('Fade Right', 'chakavak'),
                    'zoomIn' => esc_html__('Zoom In', 'chakavak'),
                    'bounceIn' => esc_html__('Bounce In', 'chakavak'),
                    'swing' => esc_html__('Swing', 'chakavak'),
                    'flash' => esc_html__('Flash', 'chakavak'),
                    'shake' => esc_html__('Shake', 'chakavak'),
                ],
            ]
        );

        $this->add_control(
            'delay',
            [
                'label' => esc_html__('Delay', 'chakavak'),
                'type' => Controls_Manager::NUMBER,
                'min' => 100,
                'max' => 10000,
                'step' => 100,
                'default' => '2000'
            ]
        );

        $this->add_control(
            'typing_speed',
            [
                'label' => esc_html__('Typing Speed', 'chakavak'),
                'type' => Controls_Manager::NUMBER,
                'min' => 100,
                'max' => 10000,
                'step' => 100,
                'default' => '100',
                'condition' => [
                    'animation_type' => 'typing',
                ],
            ]
        );

        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Loop', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'condition' => [
                    'animation_type' => 'typing',
                ],
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'cursor',
            [
                'label' => esc_html__('Cursor', 'chakavak'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'chakavak'),
                'label_off' => esc_html__('Hide', 'chakavak'),
                'default' => 'yes',
                'condition' => [
                    'animation_type' => 'typing',
                ],
            ]
        );

        $this->add_control(
            'select_tag',
            [
                'label' => esc_html__('Selct Tag', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => chakavak_heading(),
                'default' => 'h3',
            ]
        );

        $this->end_controls_section();

        //style
        $this->start_controls_section(
            'animated_style',
            [
                'label' => esc_html__('Animated Text', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'animated_space',
            [
                'label' => esc_html__('Spacing between', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__item' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'animated_width',
            [
                'label' => esc_html__('Width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 500,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__strings' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'animated_typography',
                'selector' => '{{WRAPPER}} .chk-animated-text__strings',
            ]
        );

        $this->add_control(
            'animated_color',
            [
                'label' => esc_html__('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__strings,
					 {{WRAPPER}} .chk-animated-text .typed-cursor' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'animated_shadow',
                'selector' => '{{WRAPPER}} .chk-animated-text__strings',
            ]
        );

        $this->add_responsive_control(
            'animated_text_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__strings' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'animated_text_radius',
            [
                'label' => esc_html__('Border Radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__strings' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'animated_text_bg',
                'types' => ['none', 'classic', 'gradient'],
                'exclude' => ['image'],
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Background', 'chakavak'),
                    ],
                ],
                'selector' => '{{WRAPPER}} .chk-animated-text__strings',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'prefix_style',
            [
                'label' => esc_html__('Prefix Text', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'animated_prefix!' => ''
                ],
            ]
        );

        $this->add_control(
            'prefix_color',
            [
                'label' => esc_html__('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__prefix' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'prefix_typography',
                'selector' => '{{WRAPPER}} .chk-animated-text__prefix',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'prefix_shadow',
                'selector' => '{{WRAPPER}} .chk-animated-text__prefix',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'suffix_style',
            [
                'label' => esc_html__('Suffix Text', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'animated_suffix!' => ''
                ],
            ]
        );

        $this->add_control(
            'suffix_color',
            [
                'label' => esc_html__('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-animated-text__suffix' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'suffix_typography',
                'selector' => '{{WRAPPER}} .chk-animated-text__suffix',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'suffix_shadow',
                'selector' => '{{WRAPPER}} .chk-animated-text__suffix',
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $animated_text = '';
        foreach ($settings['animated_texts'] as $item) {
            $animated_text .= $item['animated_text'] . '@@';
        }
        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div class="chk-animated-text"
                 data-animated-text="<?php echo esc_attr($animated_text); ?>"
                 data-anim="<?php echo esc_attr($settings['animation_type']); ?>"
                 data-speed="<?php echo esc_attr($settings['typing_speed']); ?>"
                 data-delay="<?php echo esc_attr($settings['delay']); ?>"
                 data-cursor="<?php echo esc_attr($settings['cursor']); ?>"
                 data-loop="<?php echo esc_attr($settings['loop']); ?>">
                <?php
                $this->add_render_attribute('item', 'class', [
                    'chk-animated-text__item d-inline-flex',
                    $settings['display'],
                ]);

                echo '<div ' . $this->get_render_attribute_string('item') . '>';

                if (!empty($settings['animated_prefix'])) {
                    echo '<span class="chk-animated-text__prefix">' . esc_html($settings['animated_prefix']) . '</span>';
                }

                if ('typing' == $settings['animation_type']) {
                    $this->add_render_attribute('animated-text', 'class', 'chk-animated-text__strings');
                    echo '<div class="chk-animated-text__typing">';
                    echo '<' . $settings['select_tag'] . ' ' . $this->get_render_attribute_string('animated-text') . '>';
                    echo esc_html($settings['animated_texts'][0]['animated_text']);
                    echo '</' . $settings['select_tag'] . '>';
                    echo '</div>';
                } else {
                    $this->add_render_attribute('animated-text', 'class', 'chk-animated-text__strings');
                    echo '<' . $settings['select_tag'] . ' ' . $this->get_render_attribute_string('animated-text') . '>';
                    echo esc_html(rtrim($animated_text, '@@'));
                    echo '</' . $settings['select_tag'] . '>';
                }

                if (!empty($settings['animated_suffix'])) {
                    echo '<span class="chk-animated-text__suffix">' . esc_html($settings['animated_suffix']) . '</span>';
                }

                echo '</div>';
                ?>
            </div>

        </div>
        <?php
    }

    protected function content_template()
    {
    }
}

Plugin::instance()->widgets_manager->register(new chakavak_Animated_Text());
